#!/usr/bin/env python

from gnuradio import gr, gru, eng_notation
from gnuradio import usrp
from gnuradio.eng_option import eng_option
from gnuradio import eng_notation
from gnuradio import optfir
from optparse import OptionParser
from usrpm import usrp_dbid
import time
import os.path
import sys

# required FPGA that can do 4 rx channels.


class my_graph(gr.top_block):

    def __init__(self):
        gr.top_block.__init__(self)

        parser = OptionParser (option_class=eng_option)
        #parser.add_option("-S", "--subdev", type="subdev", default=(0, None),
        #                  help="select USRP Rx side A or B (default=A)")
        parser.add_option("-d", "--decim", type="int", default=128,
                          help="set fgpa decimation rate to DECIM [default=%default]")
        parser.add_option("-f", "--freq", type="eng_float", default=96.8e6,
                          help="set frequency to FREQ [default=%default])", metavar="FREQ")
        parser.add_option("-g", "--gain", type="eng_float", default=45,
                          help="set gain in dB [default=%default]")
        parser.add_option("-s", "--seconds", type="eng_float", default=4.0,
                          help="Number of seconds to capture [default=%default]")
        parser.add_option("-F", "--filter", action="store_true", default=False,
                          help="Enable channel filter")
        parser.add_option("-o", "--output", type="string", default=None,
                          help="set output basename")
        (options, args) = parser.parse_args()

        if len(args) != 0:
            parser.print_help()
            raise SystemExit

        if options.output is None:
            parser.print_help()
            sys.stderr.write("You must provide an output filename base with -o OUTPUT\n")
            raise SystemExit
        else:
            basename = options.output

        nchan = 2
        nsecs = options.seconds
        if options.output_shorts:
          self.sizeof_sample = gr.sizeof_short*2
          self.typeof_sample = 'cshort'
        else:
          self.sizeof_sample = gr.sizeof_gr_complex
          self.typeof_sample = 'cfloat'

        if options.filter:
            sw_decim = 1
        else:
            sw_decim = 1

        if nchan>2:
          if options.output_shorts:
            self.u = usrp.source_c(0, options.decim, fpga_filename="std_4rx_0tx.rbf")
          else:
            self.u = usrp.source_s(0, options.decim, fpga_filename="std_4rx_0tx.rbf")
        else:
          if options.output_shorts:
            self.u = usrp.source_s(0,options.decim)
          else:
            self.u = usrp.source_c(0,options.decim)
        if options.width_8:
            sample_width = 8
            sample_shift = 8
            format = self.u.make_format(sample_width, sample_shift)
            r = self.u.set_format(format)

        if self.u.nddcs() < nchan:
            sys.stderr.write('This code requires an FPGA build with %d DDCs.  This FPGA has only %d.\n' % (
                nchan, self.u.nddcs()))
            raise SystemExit
                             
        if not self.u.set_nchannels(nchan):
            sys.stderr.write('set_nchannels(%d) failed\n' % (nchan,))
            raise SystemExit
        
        input_rate = self.u.adc_freq() / self.u.decim_rate()
        print "USB data rate   = %s" % (eng_notation.num_to_str(input_rate),)
        sink_data_rate = input_rate/sw_decim
        print "File data rate = %s" % (eng_notation.num_to_str(sink_data_rate),)

        self.subdev = self.u.db(0) + self.u.db(1)

        #if (len(self.subdev) < 4 or
        #    self.u.db(0, 0).dbid() != usrp_dbid.BASIC_RX or
        #    self.u.db(1, 0).dbid() != usrp_dbid.BASIC_RX):
        #    sys.stderr.write('This code requires a Basic Rx board on Sides A & B\n')
        #    sys.exit(1)

        #self.u.set_mux(gru.hexint(0xf3f2f1f0))
        if 4==nchan:
          self.u.set_mux(gru.hexint(0xf3f2f1f0))
        else: #2==nchan 
          self.u.set_mux(gru.hexint(0xf0f0f2f0))

        # collect 1 second worth of data
        limit = int(nsecs * input_rate * nchan)
        print "limit = ", limit
        head = gr.head(self.sizeof_sample, limit)

        # deinterleave four channels from FPGA
        di = gr.deinterleave(self.sizeof_sample)

        self.connect(self.u, head, di)
        
        # taps for channel filter
        chan_filt_coeffs = optfir.low_pass (1,           # gain
                                            input_rate,  # sampling rate
                                            80e3,        # passband cutoff
                                            115e3,       # stopband cutoff
                                            0.1,         # passband ripple
                                            60)          # stopband attenuation
        #print len(chan_filt_coeffs)

        for i in range(nchan):

            sink = gr.file_sink(self.sizeof_sample,
                                basename + ("-%s-%d-%s.dat" % (eng_notation.num_to_str(sink_data_rate), i,self.typeof_sample)))
            if options.filter:
                chan_filt = gr.fir_filter_ccf(sw_decim, chan_filt_coeffs)
                self.connect((di, i), chan_filt, sink)
            else:
                self.connect((di, i), sink)


        self.set_gain(options.gain)
        self.set_freq(options.freq)

    def set_gain(self, gain):
        for i in range(len(self.subdev)):
            self.subdev[i].set_gain(gain)

    def set_freq(self, target_freq):
        ok = True
        for i in range(len(self.subdev)):
            r = usrp.tune(self.u, i, self.subdev[i], target_freq)
            if not r:
                ok = False
                print "set_freq: failed to set subdev[%d] freq to %f" % (
                    i, target_freq)

        return ok


def main ():
    my_graph().run()

if __name__ == '__main__':
    main ()
