/* -*- c++ -*- */
/*
 * Copyright 2005 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <iostream>
#include <string>
#include <fstream>
#include <unistd.h>
#include <stdlib.h>
#include <gr_complex.h>
#include <getopt.h>
#include <gr_misc.h>
#include <limits>
#include <gr_fxpt_nco.h>
#include "time_series.h"
#include "simulation.h"

static const double C = 3e8;	// sped of light, m/s


// ------------------------------------------------------------------------

class delay_line {
  std::vector<gr_complex>	d_z;
  const int			d_mask;
  int				d_newest;
public:
  delay_line(unsigned int max_delay)
    : d_z(gr_rounduppow2(max_delay)), d_mask(d_z.size()-1), d_newest(0)
  {
  }

  void
  push_item(gr_complex x)
  {
    d_newest = (d_newest - 1) & d_mask;
    d_z[d_newest] = x;
  }

  gr_complex
  ref_item(int delay) const 
  {
    return d_z[(d_newest + delay) & d_mask];
  }
};

// ------------------------------------------------------------------------

class aux_state {
public:
  dyn_object	*d_obj;
  double	 d_last_slant_range;
  gr_fxpt_nco	 d_nco;

  aux_state(dyn_object *obj) : d_obj(obj) {}
};

// ------------------------------------------------------------------------

class my_sim : public simulation
{
  FILE			 *d_output_complete;//direct + echo
  FILE			 *d_output_direct;
  FILE			 *d_output_echo;
  FILE			 *d_output1_complete;//direct + echo
  FILE			 *d_output1_direct;
  FILE			 *d_output1_echo;
  time_series		 &d_ref;
  unsigned long long	  d_pos;		// position in time series
  delay_line		  d_z;
  dyn_object		 *d_tx;		// transmitter (not moving)
  dyn_object		 *d_rx0;		// receiver (not moving)
  dyn_object		 *d_rx1;		// second receiver (not moving)
  std::vector<aux_state*> d_target;

  double		  d_baseline;		// length of baseline for rx0 in meters
  double		  d_baseline1;		// length of baseline for rx1 in meters
  double		  d_range_bin;		// meters/range_bin
  float			  d_tx_lambda;		// wavelength of tx signals in meters
  float			  d_sample_rate;
  float			  d_gain;		// linear scale factor for reflections
  float			  d_gain_directpath;		// linear scale factor for direct path
  double                  d_tx_radians_per_time_step;  //tx phase angle per sample
                                                        //calculated from tx_freq and samplerate
                                                        //These must be accurate for the phase differences of the different antenna's to be calculated


  //void my_sim::adjust_for_start_time(double start_time);
  void adjust_for_start_time(double start_time);
  
   bool write_output(gr_complex x, FILE *output)
  {
    if (output !=NULL)
      return fwrite(&x, sizeof(x), 1, output) == 1;
    else return false;
  }

public:
  my_sim(FILE *output_complete, FILE *output_direct,FILE *output_echo, 
         FILE *output1_complete, FILE *output1_direct,FILE *output1_echo,
         time_series &ref, double timestep, float sample_rate,
	 double start_time, double tx_freq, double gain_db,double gain_db_directpath);
  ~my_sim();

  bool update();
  bool run(long long nsteps);
};


my_sim::my_sim(FILE *output_complete, FILE *output_direct,FILE *output_echo, 
              FILE *output1_complete, FILE *output1_direct,FILE *output1_echo,
               time_series &ref, double timestep,
	       float sample_rate, double start_time,
	       double tx_freq, double gain_db, double   gain_db_directpath)
  : simulation(timestep),
    d_output_complete(output_complete),d_output_direct(output_direct),d_output_echo(output_echo),
    d_output1_complete(output1_complete),d_output1_direct(output1_direct),d_output1_echo(output1_echo),
    d_ref(ref), d_pos(0), d_z(1024),
    d_range_bin(C * timestep), d_tx_lambda(C/tx_freq), 
    d_sample_rate(sample_rate), d_gain(exp10(gain_db/10)),d_gain_directpath(exp10(gain_db_directpath/10))
{
  d_tx = new dyn_object(point(0,0), point(0,0), "Tx");
  d_rx0 = new dyn_object(point(45e3,0), point(0,0), "Rx0");
  d_rx1 = new dyn_object(point(45e3+d_tx_lambda/4.0,0), point(0,0), "Rx1");
  d_tx_radians_per_time_step=2*M_PI*tx_freq*timestep;

  add_object(d_tx);
  add_object(d_rx0);
  add_object(d_rx1);
  d_baseline = dyn_object::distance(*d_tx, *d_rx0);
  d_baseline1 = dyn_object::distance(*d_tx, *d_rx1);//should be d_baseline + d_tx_lambda/4

  {
    // add targets
    float aircraft_speed;
    float aircraft_angle;
    point aircraft_pos;
    dyn_object	*ac;

    // target 1
    aircraft_speed = 135; 			// m/s
    aircraft_angle = 240 * M_PI/180;
    aircraft_pos = point(55e3, 20e3);

    ac = new dyn_object(aircraft_pos,
			point(aircraft_speed * cos(aircraft_angle),
			      aircraft_speed * sin(aircraft_angle)),
			"Ac0");
    add_object(ac);
    d_target.push_back(new aux_state(ac));

    // target 2 
    aircraft_speed = 350; 			// m/s
    aircraft_angle = 0 * M_PI/180;
    aircraft_pos = point(-20e3, 60e3);

    ac = new dyn_object(aircraft_pos,
			point(aircraft_speed * cos(aircraft_angle),
			      aircraft_speed * sin(aircraft_angle)),
			"Ac1");
    add_object(ac);
    d_target.push_back(new aux_state(ac));
  }

  adjust_for_start_time(start_time);

  for (unsigned i = 0; i < d_target.size(); i++)
    d_target[i]->d_last_slant_range =
      (dyn_object::distance(*d_tx, *d_target[i]->d_obj)
       + dyn_object::distance(*d_target[i]->d_obj, *d_rx0));

}

my_sim::~my_sim()
{
}

void
my_sim::adjust_for_start_time(double start_time)
{
  for (unsigned i = 0; i < d_obj.size(); i++){
    // Adjust initial starting positions depending on simulation
    // start time.  FIXME Assumes velocity is constant
    point p = d_obj[i]->pos();
    point v = d_obj[i]->vel();
    p.set_x(p.x() + v.x() * start_time);
    p.set_y(p.y() + v.y() * start_time);
    d_obj[i]->set_pos(p);
  }
}

bool
my_sim::update()
{
  // std::cout << *d_ac0 << std::endl;

  // grab new item from input and insert it into delay line
  const gr_complex *in = (const gr_complex *) d_ref.seek(d_pos++, 1);
  if (in == 0)
    return false;
  d_z.push_item(*in);

  gr_complex s_direct = 0;//rx0 signal contribution from direct path
  gr_complex s_echo=0;	// rx0 signal contribution from echo
  gr_complex s1_direct;//rx1 signal contribution from direct path
  gr_complex s1_echo=0;	// rx1 signal contribution from echo
  // FIXME ought to add in attenuated direct path input
  s_direct =  d_gain_directpath *d_z.ref_item(0);//(attenuated) direct path 
  double dbl_rx0_rx1_delay =(d_baseline1 - d_baseline) / d_range_bin;
  int int_rx0_rx1_delay = lrint(dbl_rx0_rx1_delay);//probably zero
  //fprintf(stderr, "int_rx0_rx1_delay %i\n", int_rx0_rx1_delay);
  double rx0_rx1_delay_delay_error = dbl_rx0_rx1_delay - int_rx0_rx1_delay;
  double rx0_rx1_phase_error=rx0_rx1_delay_delay_error*d_tx_radians_per_time_step;//allways pi/2 (90 degrees)
  //fprintf(stderr, "rx0_rx1_phase_error %f\n", rx0_rx1_phase_error*360/(2*M_PI));
  s1_direct = d_gain_directpath *d_z.ref_item(int_rx0_rx1_delay);//(attenuated) direct path to rx1 probably the same as rx0
  s1_direct = s1_direct*gr_complex(cos(rx0_rx1_phase_error),sin(rx0_rx1_phase_error));//compensate for phase_error from the time_steps
  // for each target, compute slant_range and slant_range'

  for (unsigned i = 0; i < d_target.size(); i++){
    aux_state *t = d_target[i];
    
    double slant_range = 
      (dyn_object::distance(*d_tx, *t->d_obj)
       + dyn_object::distance(*t->d_obj, *d_rx0));			  // meters (rx0 antenna)

    double slant_range1 = 
      (dyn_object::distance(*d_tx, *t->d_obj)
       + dyn_object::distance(*t->d_obj, *d_rx1));			  // meters (rx1 antenna)

    double delta_slant_range = slant_range - t->d_last_slant_range;
    t->d_last_slant_range = slant_range;
    double deriv_slant_range_wrt_time = delta_slant_range / timestep();	  // m/sec
    //We assume antenna's rx0 and rx1 are close to each other so we can ignore the doppler differences

    //fprintf(stdout, "%10.3f\t%10.3f\n", slant_range, deriv_slant_range_wrt_time);

    //do x0
    // FIXME, may want to interpolate between two bins.
    double dbl_delay =(slant_range - d_baseline) / d_range_bin;
    int int_delay = lrint(dbl_delay);
    double delay_error = dbl_delay - int_delay;
    //Now we can interpolate or calculate the phase_error and use this to compensate
    //Calculate phase-error
    double phase_error=delay_error*d_tx_radians_per_time_step;//tx_freq and samplerate have to be accurate for this to be correct
    //fprintf(stderr, "echo phase_error0 %f\n", phase_error*360/(2*M_PI));
    gr_complex x = d_z.ref_item(int_delay);
    x=x*gr_complex(cos(phase_error),sin(phase_error));//compensate for phase_error from the time_steps

    // scale amplitude (this includes everything: RCS, antenna gain, losses, etc...)
    x = x * d_gain;

    //Now do x1
    dbl_delay =(slant_range1 - d_baseline) / d_range_bin;
    int_delay = lrint(dbl_delay);
    delay_error = dbl_delay - int_delay;
    double phase_error1=delay_error*d_tx_radians_per_time_step;//tx_freq and samplerate have to be accurate for this to be correct   
    //fprintf(stderr, "echo phase_error1 %f\n", phase_error1*360/(2*M_PI));
    //fprintf(stderr, "echo phase_diff10 %f\n", (phase_error1-phase_error)*360/(2*M_PI));
    gr_complex x1 = d_z.ref_item(int_delay);//probably most of the time the same as x
    x1=x1*gr_complex(cos(phase_error1),sin(phase_error1));//This will differ from x. compensate for phase_error from the time_steps
    x1 = x1 * d_gain;

    if (1){
      // compute doppler and apply it
      float f_doppler = -deriv_slant_range_wrt_time / d_tx_lambda;

      t->d_nco.set_freq(f_doppler / d_sample_rate);
      gr_complex phasor(t->d_nco.cos(), t->d_nco.sin());
      x = x * phasor;
      x1 = x1 * phasor;//assume antenna's are close together so they will get the same doppler
      t->d_nco.step();
    }

    s_echo += x;		// add in this target's contribution
    s1_echo += x1;		// add in this target's contribution
  }
  gr_complex s_complete= s_direct+s_echo;	// complete rx0 signal =contribution from direct path and echo
  gr_complex s1_complete= s1_direct+s1_echo;	// complete rx1 signal =contribution from direct path and echo
  write_output(s_complete,d_output_complete);
  write_output(s_direct,d_output_direct);
  write_output(s_echo,d_output_echo);
  write_output(s1_complete,d_output1_complete);
  write_output(s1_direct,d_output1_direct);
  write_output(s1_echo,d_output1_echo);

  return simulation::update();		// run generic update
}

/*bool
my_sim::check(const char *rx0_filename,const char *rx0_filename,unsigned long long ref_starting_offset)
{
    d_pos=0;
    const gr_complex *ref = (const gr_complex *) d_ref.seek(d_pos, 1);//reading ref file from start

    time_series ts_rx0(sizeof(gr_complex), rx0_filename, ref_starting_offset, 0);//reading output 0 from start
    const gr_complex *rx0 = (const gr_complex *) ts_rx0.seek(d_pos, 1);//reading ref file from start
    time_series ts_rx1(sizeof(gr_complex), rx1_filename, ref_starting_offset, 0);//reading output 1 from start
    const gr_complex *rx1 = (const gr_complex *) ts_rx1.seek(d_pos, 1);//reading ref file from start
    gr_complex cross_corr=gr_complex(0.0,0.0);
    gr_complex corr1=gr_complex(0.0,0.0);
    double amp_rx0=0.0;
    double amp_rx1=0.0;
    int i,skipped;
    //determine average phase and amplitude differences between rx0 and rx1
    for(i=0;i<d_ref.nsamples_available;i++)
    {
      cross_corr+=rx0[i]*conj(rx1[i]);
      //What we actually want is corr+=rx0[i]/rx1[i]; but this fails if rx1[i] is zero;
      abs_rx1=abs(rx1[i]);
      if(abs_rx1)<1e-4)
        skipped++;
      else
        corr+=rx0[i]/rx1[i];
      //auto_corr0+=rx0[i]*conj(rx0[i]);
      //auto_corr1+=rx1[i]*conj(rx0[i]);
      amp_rx0+=abs(rx0[i]);
      amp_rx1+=abs(rx1[i]);
    }
    //int nsamples_used=d_ref.nsamples_available - skipped;
    corr1=corr1/(d_ref.nsamples_available - skipped);
    cross_corr=conj(cross_corr)/d_ref.nsamples_available;
    amp_rx0=amp_rx0/d_ref.nsamples_available;
    amp_rx1=amp_rx1/d_ref.nsamples_available;
    double amp_cross_corr=abs(cross_corr);
    double phase_cross_corr=atan2(cross_corr);
    fprintf("phase_cross_corr in degrees=%f\n",360.0*phase_cross_corr/M_PI);//should be 90
    fprintf("amp_rx1/amp_rx0=%f\n",amp_rx1/amp_rx0);
    fprintf("amp_cross-corr=%f\n",amp_cross_corr);
    fprintf("factor=%f\n",amp_cross_corr/(amp_rx1/amp_rx0)); 
    fprintf("diff=%f\n",amp_cross_corr-(amp_rx1/amp_rx0)); 
    gr_complex corr2=(amp_rx1/amp_rx0)*cross_corr/(amp_cross_corr);
    //gr_complex corr=(amp_rx1/amp_rx0)*polar(1.0,atan2(cross_corr));//alternative implementation
    for(i=0;i<d_ref.nsamples_available;i++)
    {
      rx1_corr=corr2*rx1[i];
      //rx1_corr=corr1*rx1[i];
      //rx1_corr=rx1[i]/cross_corr;
      approximate_ref=2*rx0[i]-rx1_corr;
      approximate_echo=rx0[i]-rx1_corr;//This has a complicated distorted phase component dependant on both direction of tx as direction of echo('s)
    }
//    fseek (d_output,0,SEEK_SET);//reading back output file from start
//    fseek (d_output1,0,SEEK_SET);//reading back second output file from start
//    block_size=
//    for(int i=0;i<d_ref.nsamples_available;i+=block_size)
//    {
//      int  fread (void * buffer, size_t size, size_t count, FILE * stream); 
//    }
}*/

bool
my_sim::run(long long nsteps)
{
  //fprintf(stdout, "<%12.2f, %12.2f>\n", d_ac0->pos().x(), d_ac0->pos().y());
  //std::cout << *d_ac0 << std::endl;
  bool ok = simulation::run(nsteps);
  //std::cout << *d_ac0 << std::endl;
  //fprintf(stdout, "<%12.2f, %12.2f>\n", d_ac0->pos().x(), d_ac0->pos().y());
  return ok;
}

// ------------------------------------------------------------------------

static void
usage(const char *argv0)
{
  const char *progname;
  const char *t = std::strrchr(argv0, '/');
  if (t != 0)
    progname = t + 1;
  else
    progname = argv0;
    
  fprintf(stderr, "usage: %s [options] ref_file\n", progname);
  fprintf(stderr, "    -o OUTPUT0_FILENAME [default=sim0.dat]\n");
  fprintf(stderr, "    -O OUTPUT1_FILENAME [default=sim1.dat]\n");
  fprintf(stderr, "    -p OUTPUT0_DIRECT_FILENAME [default=sim0_direct.dat]\n");
  fprintf(stderr, "    -P OUTPUT1_DIRECT_FILENAME [default=sim1_direct.dat]\n");
  fprintf(stderr, "    -q OUTPUT0_ECHO_FILENAME [default=sim0_echo.dat]\n");
  fprintf(stderr, "    -Q OUTPUT1_ECHO FILENAME [default=sim1_echo.dat]\n");
  fprintf(stderr, "    -n NSAMPLES_TO_PRODUCE [default=+inf]\n");
  fprintf(stderr, "    -s NSAMPLES_TO_SKIP [default=0]\n");
  fprintf(stderr, "    -g reflection gain in dB (should be <= 0) [default=0]\n");
  fprintf(stderr, "    -G direct path gain in dB (should be <= 0) [default=0]\n");
  fprintf(stderr, "    -f transmitter freq in Hz [default=100MHz]\n");
  fprintf(stderr, "    -r sample rate in Hz [default=250kHz]\n");
  fprintf(stderr, "    -S simulation start time in seconds [default=0]\n");
}

int
main(int argc, char **argv)
{
  int	ch;
  const char *output_complete_filename = "sim0.dat";
  const char *output_direct_filename = "sim0_direct.dat";
  const char *output_echo_filename = "sim0_echo.dat";
  const char *output1_complete_filename = "sim1.dat";
  const char *output1_direct_filename = "sim1_direct.dat";
  const char *output1_echo_filename = "sim1_echo.dat";
  const char *ref_filename = 0;
  long long int nsamples_to_skip = 0;
  long long int nsamples_to_produce = std::numeric_limits<long long int>::max();
  double sample_rate = 250e3;
  double gain_db = 0;
  double gain_db_directpath = 0;
  double tx_freq = 100e6;
  double start_time = 0;

  while ((ch = getopt(argc, argv, "o:O:p:P:q:Q:s:n:g:G:f:S:")) != -1){
    switch (ch){
    case 'o':
      output_complete_filename = optarg;
      break;

    case 'O':
      output1_complete_filename = optarg;
      break;
    case 'p':
      output_direct_filename = optarg;
      break;
    case 'P':
      output1_direct_filename = optarg;
      break;
    case 'q':
      output_echo_filename = optarg;
      break;
    case 'Q':
      output1_echo_filename = optarg;
      break;
      
    case 's':
      nsamples_to_skip = (long long) strtof(optarg, 0);
      if (nsamples_to_skip < 0){
	usage(argv[0]);
	fprintf(stderr, "    nsamples_to_skip must be >= 0\n");
	exit(1);
      }
      break;

    case 'n':
      nsamples_to_produce = (long long) strtof(optarg, 0);
      if (nsamples_to_produce < 0){
	usage(argv[0]);
	fprintf(stderr, "    nsamples_to_produce must be >= 0\n");
	exit(1);
      }
      break;

    case 'g':
      gain_db = strtof(optarg, 0);
      break;

    case 'G':
      gain_db_directpath = strtof(optarg, 0);
      break;

    case 'f':
      tx_freq = strtof(optarg, 0);
      break;

    case 'r':
      sample_rate = strtof(optarg, 0);
      break;

    case 'S':
      start_time = strtof(optarg, 0);
      break;

    case '?':
    case 'h':
    default:
      usage(argv[0]);
      exit(1);
    }
  } // while getopt

  if (argc - optind != 1){
    usage(argv[0]);
    exit(1);
  }

  ref_filename = argv[optind++];

  double timestep = 1.0/sample_rate;


  FILE *output_complete = fopen(output_complete_filename, "wb");//"wb"
  if (output_complete == 0){
    perror(output_complete_filename);
    exit(1);
  }
  FILE *output_direct = fopen(output_direct_filename, "wb");//"wb"
  if (output_direct == 0){
    perror(output_direct_filename);
    exit(1);
  }
  FILE *output_echo = fopen(output_echo_filename, "wb");//"wb"
  if (output_echo == 0){
    perror(output_echo_filename);
    exit(1);
  }
  FILE *output1_complete = fopen(output1_complete_filename, "wb");//"wb"
  if (output1_complete == 0){
    perror(output1_complete_filename);
    exit(1);
  }
  FILE *output1_direct = fopen(output1_direct_filename, "wb");//"wb"
  if (output1_direct == 0){
    perror(output1_direct_filename);
    exit(1);
  }
  FILE *output1_echo = fopen(output1_echo_filename, "wb");//"wb"
  if (output1_echo == 0){
    perror(output1_echo_filename);
    exit(1);
  }


  unsigned long long ref_starting_offset = 0;
  ref_starting_offset += nsamples_to_skip;

  try {
    time_series ref(sizeof(gr_complex), ref_filename, ref_starting_offset, 0);

    my_sim	simulator(output_complete,output_direct,output_echo,
                          output1_complete,output1_direct,output1_echo,
                           ref, timestep, sample_rate, start_time,
			  tx_freq, gain_db,gain_db_directpath);
    simulator.run(nsamples_to_produce);
  }
  catch (std::string &s){
    std::cerr << s << std::endl;
    exit(1);
  }

  return 0;
}

