/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gri_fft_double.h>
#include <fftw3.h>
#include <gr_complex.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <cassert>
#include <stdexcept>

static char *
wisdom_filename_double ()
{
  static char *filename = ".gr_radar_fftw_double_wisdom";

  char	*home = getenv ("HOME");
  if (home){
    char *p = new char[strlen (home) + strlen (filename) + 2];
    strcpy (p, home);
    strcat (p, "/");
    strcat (p, filename);
    return p;
  }
  return 0;
}

static void 
gri_fftw_double_import_wisdom ()
{
  char *filename = wisdom_filename_double ();
  FILE *fp = fopen (filename, "r");
  if (fp != 0){
    int r = fftw_import_wisdom_from_file (fp);
    fclose (fp);
    if (!r){
      fprintf (stderr, "gri_fftw_double: can't import wisdom from %s\n", filename);
    }
  }
  delete [] filename;
}

static void
gri_fftw_double_export_wisdom ()
{
  char *filename = wisdom_filename_double ();
  FILE *fp = fopen (filename, "w");
  if (fp != 0){
    fftw_export_wisdom_to_file (fp);
    fclose (fp);
  }
  else {
    fprintf (stderr, "gri_fftw_double: ");
    perror (filename);
  }
  delete [] filename;
}

// ----------------------------------------------------------------

gri_fft_complexd::gri_fft_complexd (int fft_size, bool forward)
{
  assert (sizeof (fftw_complex) == sizeof (gr_complexd));
  
  if (fft_size <= 0)
    throw std::out_of_range ("gri_fftw_double: invalid fft_size");
  
  d_fft_size = fft_size;
  d_inbuf = (gr_complexd *) fftw_malloc (sizeof (gr_complexd) * inbuf_length ());
  if (d_inbuf == 0)
    throw std::runtime_error ("fftw_malloc");
  
  d_outbuf = (gr_complexd *) fftw_malloc (sizeof (gr_complexd) * outbuf_length ());
  if (d_outbuf == 0){
    fftw_free (d_inbuf);
    throw std::runtime_error ("fftw_malloc");
  }

  // FIXME If there's ever a chance that the planning functions
  // will be called in multiple threads, we've got to ensure single
  // threaded access.  They are not thread-safe.
  
  gri_fftw_double_import_wisdom ();	// load prior wisdom from disk
  d_plan = fftw_plan_dft_1d (fft_size,
			      reinterpret_cast<fftw_complex *>(d_inbuf), 
			      reinterpret_cast<fftw_complex *>(d_outbuf),
			      forward ? FFTW_FORWARD : FFTW_BACKWARD,
			      FFTW_MEASURE);

  if (d_plan == NULL) {
    fprintf(stderr, "gri_fft_complexd: error creating plan\n");
    throw std::runtime_error ("fftw_plan_dft_1d failed");
  }
  gri_fftw_double_export_wisdom ();	// store new wisdom to disk
}

gri_fft_complexd::~gri_fft_complexd ()
{
  fftw_destroy_plan ((fftw_plan) d_plan);
  fftw_free (d_inbuf);
  fftw_free (d_outbuf);
}

void
gri_fft_complexd::execute ()
{
  fftw_execute ((fftw_plan) d_plan);
}

// ----------------------------------------------------------------

gri_fft_real_fwd_double::gri_fft_real_fwd_double (int fft_size)
{
  assert (sizeof (fftw_complex) == sizeof (gr_complexd));
  
  if (fft_size <= 0)
    throw std::out_of_range ("gri_fftw_double: invalid fft_size");
  
  d_fft_size = fft_size;
  d_inbuf = (double *) fftw_malloc (sizeof (double) * inbuf_length ());
  if (d_inbuf == 0)
    throw std::runtime_error ("fftw_malloc");
  
  d_outbuf = (gr_complexd *) fftw_malloc (sizeof (gr_complexd) * outbuf_length ());
  if (d_outbuf == 0){
    fftw_free (d_inbuf);
    throw std::runtime_error ("fftw_malloc");
  }

  // FIXME If there's ever a chance that the planning functions
  // will be called in multiple threads, we've got to ensure single
  // threaded access.  They are not thread-safe.
  
  gri_fftw_double_import_wisdom ();	// load prior wisdom from disk
  d_plan = fftw_plan_dft_r2c_1d (fft_size,
				  d_inbuf,
				  reinterpret_cast<fftw_complex *>(d_outbuf),
				  FFTW_MEASURE);

  if (d_plan == NULL) {
    fprintf(stderr, "gri_fft_real_fwd_double: error creating plan\n");
    throw std::runtime_error ("fftw_plan_dft_r2c_1d failed");
  }
  gri_fftw_double_export_wisdom ();	// store new wisdom to disk
}

gri_fft_real_fwd_double::~gri_fft_real_fwd_double ()
{
  fftw_destroy_plan ((fftw_plan) d_plan);
  fftw_free (d_inbuf);
  fftw_free (d_outbuf);
}

void
gri_fft_real_fwd_double::execute ()
{
  fftw_execute ((fftw_plan) d_plan);
}

// ----------------------------------------------------------------

gri_fft_real_rev_double::gri_fft_real_rev_double (int fft_size)
{
  assert (sizeof (fftw_complex) == sizeof (gr_complexd));
  
  if (fft_size <= 0)
    throw std::out_of_range ("gri_fftw_double: invalid fft_size");
  
  d_fft_size = fft_size;
  d_inbuf = (gr_complexd *) fftw_malloc (sizeof (gr_complexd) * inbuf_length ());
  if (d_inbuf == 0)
    throw std::runtime_error ("fftw_malloc");
  
  d_outbuf = (double *) fftw_malloc (sizeof (double) * outbuf_length ());
  if (d_outbuf == 0){
    fftw_free (d_inbuf);
    throw std::runtime_error ("fftw_malloc");
  }

  // FIXME If there's ever a chance that the planning functions
  // will be called in multiple threads, we've got to ensure single
  // threaded access.  They are not thread-safe.
  
  gri_fftw_double_import_wisdom ();	// load prior wisdom from disk
  d_plan = fftw_plan_dft_c2r_1d (fft_size,
				  reinterpret_cast<fftw_complex *>(d_inbuf),
				  d_outbuf,
				  FFTW_MEASURE);

  if (d_plan == NULL) {
    fprintf(stderr, "gri_fft_real_rev_double: error creating plan\n");
    throw std::runtime_error ("fftw_plan_dft_c2r_1d failed");
  }
  gri_fftw_double_export_wisdom ();	// store new wisdom to disk
}

gri_fft_real_rev_double::~gri_fft_real_rev_double ()
{
  fftw_destroy_plan ((fftw_plan) d_plan);
  fftw_free (d_inbuf);
  fftw_free (d_outbuf);
}

void
gri_fft_real_rev_double::execute ()
{
  fftw_execute ((fftw_plan) d_plan);
}

